<?php

namespace App\Services;

use App\Models\Cashbook;
use App\Models\Payment;
use App\Models\Sale;
use App\Models\Purchase;
use App\Models\Expense;
use App\Models\Income;
use App\Models\Returns;
use App\Models\ReturnPurchase;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Exception;

class CashbookService
{
    /**
     * Create a cashbook entry from a payment
     * 
     * @param Payment $payment
     * @return Cashbook|null
     */
    public function createFromPayment(Payment $payment)
    {
        try {
            // Only process cash payments
            if (!$this->isCashPayment($payment->paying_method)) {
                return null;
            }

            // Determine if this is a sale payment (debit) or purchase payment (credit)
            $type = $payment->sale_id ? 'debit' : 'credit';
            $details = $this->generatePaymentDetails($payment);
            
            return $this->createEntry([
                'date' => $payment->payment_at ?? now(),
                'type' => $type,
                'details' => $details,
                'amount' => $payment->amount,
                'warehouse_id' => $this->getWarehouseFromPayment($payment),
                'user_id' => $payment->user_id ?? auth()->id(),
                'note' => $payment->payment_note,
                'reference_type' => get_class($payment),
                'reference_id' => $payment->id,
            ]);
        } catch (Exception $e) {
            Log::error('Failed to create cashbook entry from payment: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Create a cashbook entry from an expense
     * 
     * @param Expense $expense
     * @return Cashbook|null
     */
    public function createFromExpense(Expense $expense)
    {
        try {
            return $this->createEntry([
                'date' => $expense->created_at,
                'type' => 'credit',
                'details' => 'Expense: ' . ($expense->expenseCategory->name ?? 'General') . ' - ' . $expense->reference_no,
                'amount' => $expense->amount,
                'warehouse_id' => $expense->warehouse_id,
                'user_id' => $expense->user_id ?? auth()->id(),
                'note' => $expense->note,
                'reference_type' => get_class($expense),
                'reference_id' => $expense->id,
            ]);
        } catch (Exception $e) {
            Log::error('Failed to create cashbook entry from expense: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Create a cashbook entry from an income
     * 
     * @param Income $income
     * @return Cashbook|null
     */
    public function createFromIncome(Income $income)
    {
        try {
            return $this->createEntry([
                'date' => $income->created_at,
                'type' => 'debit',
                'details' => 'Income: ' . ($income->incomeCategory->name ?? 'General') . ' - ' . $income->reference_no,
                'amount' => $income->amount,
                'warehouse_id' => $income->warehouse_id,
                'user_id' => $income->user_id ?? auth()->id(),
                'note' => $income->note,
                'reference_type' => get_class($income),
                'reference_id' => $income->id,
            ]);
        } catch (Exception $e) {
            Log::error('Failed to create cashbook entry from income: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Create a cashbook entry from a sale return (credit - money going out)
     * 
     * @param Returns $return
     * @return Cashbook|null
     */
    public function createFromSaleReturn(Returns $return)
    {
        try {
            $sale = $return->sale;
            if (!$sale) {
                return null;
            }

            return $this->createEntry([
                'date' => $return->created_at,
                'type' => 'credit',
                'details' => 'Sale Return: ' . $sale->reference_no . ' - Refund to customer',
                'amount' => $return->grand_total,
                'warehouse_id' => $return->warehouse_id,
                'user_id' => $return->user_id ?? auth()->id(),
                'note' => 'Return for sale: ' . $sale->reference_no,
                'reference_type' => get_class($return),
                'reference_id' => $return->id,
            ]);
        } catch (Exception $e) {
            Log::error('Failed to create cashbook entry from sale return: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Create a cashbook entry from a purchase return (debit - money coming in)
     * 
     * @param ReturnPurchase $return
     * @return Cashbook|null
     */
    public function createFromPurchaseReturn(ReturnPurchase $return)
    {
        try {
            $purchase = $return->purchase;
            if (!$purchase) {
                return null;
            }

            return $this->createEntry([
                'date' => $return->created_at,
                'type' => 'debit',
                'details' => 'Purchase Return: ' . $purchase->reference_no . ' - Refund from supplier',
                'amount' => $return->grand_total,
                'warehouse_id' => $return->warehouse_id,
                'user_id' => $return->user_id ?? auth()->id(),
                'note' => 'Return for purchase: ' . $purchase->reference_no,
                'reference_type' => get_class($return),
                'reference_id' => $return->id,
            ]);
        } catch (Exception $e) {
            Log::error('Failed to create cashbook entry from purchase return: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Core method to create a cashbook entry with balance calculation
     * 
     * @param array $data
     * @return Cashbook
     */
    public function createEntry(array $data)
    {
        return DB::transaction(function () use ($data) {
            // Check if entry already exists (prevent duplicates)
            if (isset($data['reference_type']) && isset($data['reference_id'])) {
                $existing = Cashbook::where('reference_type', $data['reference_type'])
                    ->where('reference_id', $data['reference_id'])
                    ->first();
                
                if ($existing) {
                    Log::info('Cashbook entry already exists for this reference');
                    return $existing;
                }
            }

            // Ensure auto-generated entries are marked as non-manual
            if (!isset($data['is_manual'])) {
                $data['is_manual'] = false;
            }

            // Create the entry
            $cashbook = Cashbook::create($data);

            // Recalculate balances for this warehouse
            Cashbook::recalculateBalances($data['warehouse_id'] ?? null);

            return $cashbook;
        });
    }

    /**
     * Update cashbook entry when source record is updated
     * 
     * @param string $referenceType
     * @param int $referenceId
     * @param array $data
     * @return bool
     */
    public function updateEntry(string $referenceType, int $referenceId, array $data)
    {
        try {
            $cashbook = Cashbook::where('reference_type', $referenceType)
                ->where('reference_id', $referenceId)
                ->first();

            if ($cashbook) {
                $warehouse_id = $cashbook->warehouse_id;
                $cashbook->update($data);
                Cashbook::recalculateBalances($warehouse_id);
                return true;
            }

            return false;
        } catch (Exception $e) {
            Log::error('Failed to update cashbook entry: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Delete cashbook entry when source record is deleted
     * 
     * @param string $referenceType
     * @param int $referenceId
     * @return bool
     */
    public function deleteEntry(string $referenceType, int $referenceId)
    {
        try {
            $cashbook = Cashbook::where('reference_type', $referenceType)
                ->where('reference_id', $referenceId)
                ->first();

            if ($cashbook) {
                $warehouse_id = $cashbook->warehouse_id;
                $cashbook->delete();
                Cashbook::recalculateBalances($warehouse_id);
                return true;
            }

            return false;
        } catch (Exception $e) {
            Log::error('Failed to delete cashbook entry: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Check if payment method is cash
     * 
     * @param string $method
     * @return bool
     */
    protected function isCashPayment(string $method): bool
    {
        $cashMethods = ['cash', 'Cash'];
        return in_array($method, $cashMethods);
    }

    /**
     * Generate payment details description
     * 
     * @param Payment $payment
     * @return string
     */
    protected function generatePaymentDetails(Payment $payment): string
    {
        if ($payment->sale_id) {
            $sale = Sale::find($payment->sale_id);
            $customer = $sale ? $sale->customer : null;
            return 'Payment received - Sale #' . ($sale->reference_no ?? $payment->sale_id) . 
                   ($customer ? ' - ' . $customer->name : '');
        }

        if ($payment->purchase_id) {
            $purchase = Purchase::find($payment->purchase_id);
            $supplier = $purchase ? $purchase->supplier : null;
            return 'Payment made - Purchase #' . ($purchase->reference_no ?? $payment->purchase_id) . 
                   ($supplier ? ' - ' . $supplier->name : '');
        }

        return 'Payment - Ref: ' . ($payment->payment_reference ?? $payment->id);
    }

    /**
     * Get warehouse from payment
     * 
     * @param Payment $payment
     * @return int|null
     */
    protected function getWarehouseFromPayment(Payment $payment): ?int
    {
        if ($payment->sale_id) {
            $sale = Sale::find($payment->sale_id);
            return $sale ? $sale->warehouse_id : null;
        }

        if ($payment->purchase_id) {
            $purchase = Purchase::find($payment->purchase_id);
            return $purchase ? $purchase->warehouse_id : null;
        }

        return null;
    }

    /**
     * Get opening balance for a warehouse
     * 
     * @param int|null $warehouseId
     * @return float
     */
    public function getOpeningBalance(?int $warehouseId = null): float
    {
        $query = Cashbook::orderBy('date', 'asc')->orderBy('id', 'asc');
        
        if ($warehouseId) {
            $query->where('warehouse_id', $warehouseId);
        }

        $firstEntry = $query->first();
        return $firstEntry ? $firstEntry->balance : 0;
    }

    /**
     * Get current balance
     * 
     * @param int|null $warehouseId
     * @return float
     */
    public function getCurrentBalance(?int $warehouseId = null): float
    {
        $query = Cashbook::orderBy('date', 'desc')->orderBy('id', 'desc');
        
        if ($warehouseId) {
            $query->where('warehouse_id', $warehouseId);
        }

        $latestEntry = $query->first();
        return $latestEntry ? $latestEntry->balance : 0;
    }
}
